<?php
/**
 * Passkey integration for Really Simple SSL Pro.
 *
 * Handles registration of scripts, the onboarding flow, and REST controller setup
 * for passkey-based two-factor authentication.
 *
 * @package   ReallySimpleSSL\Pro\Security\WordPress\Passkey
 * @author    Really Simple Plugins
 * @link      https://really-simple-ssl.com
 */

namespace RSSSL\Pro\Security\WordPress\Passkey;

use RSSSL\Pro\Security\WordPress\Passkey\Models\Rsssl_Webauthn;
use RSSSL\Pro\Security\WordPress\Two_Fa\Providers\Rsssl_Two_Factor_Passkey;
use RSSSL\Security\WordPress\Two_Fa\Controllers\Rsssl_Base_Controller;
use RSSSL\Security\WordPress\Two_Fa\Rsssl_Two_Fa_Authentication;
use RSSSL\Security\WordPress\Two_Fa\Rsssl_Two_Factor;
use RSSSL\Security\WordPress\Two_Fa\Traits\Rsssl_Two_Fa_Helper;
use WP_User;

/**
 * Bootstraps passkey-based two-factor authentication.
 *
 * - Registers enqueue scripts and login hooks
 * - Triggers onboarding UI if needed
 * - Delegates to the REST & base controllers
 *
 * @package ReallySimpleSSL\Pro\Security\WordPress\Passkey
 */
class Rsssl_Passkey {
	use Rsssl_Two_Fa_Helper;

	private const PLUGIN_SLUG = 'really-simple-security';
	private const API_VERSION = 'v1';
	private const TWO_FA_VERSION = 'v2';
	public const REST_NAMESPACE = self::PLUGIN_SLUG . '/' . self::API_VERSION . '/two-fa/' . self::TWO_FA_VERSION;

	private static ?self $instance = null;

	/**  Holds the user who’s in the middle of onboarding  */
	private static ?WP_User $onboarding_user = null;


	/**
	 * Retrieve single instance.
	 *
	 * @return self
	 */
	public static function get_instance(): self {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Run hooks
	 * @return void
	 */
	public static function run_hooks(): void {
		if ( ( defined( 'RSSSL_DISABLE_2FA' ) && RSSSL_DISABLE_2FA )
		     || ( defined( 'RSSSL_SAFE_MODE' ) && RSSSL_SAFE_MODE )
		) {
			return;
		}
		// if 2fa is enabled, we need to run the hooks. Since this is a feature of the two-factor authentication, we check if the login protection is enabled.
		// if so 2fa takes over and this is not needed.
		$instance = self::get_instance();
		if ( rsssl_get_option('login_protection_enabled', false) ) {
			// we only load the scripts if the login protection is enabled.
			// enqueue our CSS/JS on the WP login screen
			add_action( 'login_enqueue_scripts', [ $instance, 'enqueue_onboarding_scripts' ] );
			// enqueue our CSS on the WP login screen
			add_action( 'login_enqueue_scripts', [ $instance, 'enqueue_onboarding_styles' ] );
			return;
		}


		// Check if the passkey table is enabled.
		add_action( 'init', [ Rsssl_Webauthn::class, 'maybe_install_table' ] );
		// enqueue our CSS/JS on the WP login screen
		add_action( 'login_enqueue_scripts', [ $instance, 'enqueue_onboarding_scripts' ] );
		// enqueue our CSS on the WP login screen
		add_action( 'login_enqueue_scripts', [ $instance, 'enqueue_onboarding_styles' ] );
		// after successful login, maybe kick off the passkey setup
		add_action( 'wp_login', [ $instance, 'maybe_start_passkey_onboarding' ], 10, 2 );
		/*
		 * Start the controller for the passkey this is needed for all the logic to work.
		 */
		Rsssl_Two_Factor_Passkey::start_controller( self::PLUGIN_SLUG, self::API_VERSION, self::TWO_FA_VERSION );
		// We add the skip functionality to the two-factor authentication.
		new Rsssl_Base_Controller( self::PLUGIN_SLUG, self::API_VERSION, self::TWO_FA_VERSION );

		if ( rsssl_admin_logged_in() ) {
			( new RSSSL_Passkey_User_Admin() );
		}

		// We add the passkey profile settings to the user profile.
		Rsssl_Passkey_Profile_Settings::init();
	}

	/**
	 * Enqueue onboarding styles
	 * @return void
	 */
	public function enqueue_onboarding_styles(): void {
		$uri       = trailingslashit( rsssl_url ) . 'assets/features/two-fa/styles.min.css';
		$file_path = trailingslashit( rsssl_path ) . 'assets/features/two-fa/styles.min.css';

		if ( file_exists( $file_path ) ) {
			wp_enqueue_style( 'rsssl-passkey-settings', $uri, [], filemtime( $file_path ) );
		}
	}

	/**
	 * Enqueue onboarding scripts
	 * @return void
	 */
	public function enqueue_onboarding_scripts(): void {
		$uri       = trailingslashit( rsssl_url ) . 'assets/features/two-fa/assets.min.js';
		$file_path = trailingslashit( rsssl_path ) . 'assets/features/two-fa/assets.min.js';
		if ( file_exists( $file_path ) ) {
			// The reason this now has a fallback is that the script is loaded as a module script,
			// which is not supported in all browsers. also when using wp_enqueue_script_module it totally broke
			// the login page. So we need to load it as a normal script and then add the type="module" to the script tag.
			$this->fallback_enqueue_script( $uri, $file_path );

			// we get the user_login from the post data, so we can use it in the script.
			$user_login = isset( $_POST['log'] ) ? sanitize_user( wp_unslash( $_POST['log'] ) ) : '';

			// now get the id from the user_login
			$user             = get_user_by( 'login', $user_login );
			$verified_user_id = $user->ID ?? 0;
			add_filter( 'rsssl_two_factor_translatables', [ Rsssl_Two_Factor_Passkey::class, 'translatables' ] );
			$login_nonce = '';
			if ( (bool) rsssl_get_option('login_protection_enabled', false) === false && $verified_user_id > 0 ) {
				// Create a nonce for the user to use in the passkey onboarding.
				$login_nonce = Rsssl_Two_Fa_Authentication::create_login_nonce( $verified_user_id )['rsssl_key'];
			}

			wp_localize_script( 'rsssl-passkey-login', 'rsssl_login', [
				'nonce'         => wp_create_nonce( 'wp_rest_passkey_onboarding' ),
				'origin'        => 'passkey',
				'root'          => esc_url_raw( rest_url( self::REST_NAMESPACE ) ),
				'login_nonce'   => $login_nonce,
				'redirect_to'   => isset( $_REQUEST['redirect_to'] ) ? esc_url_raw( wp_unslash( $_REQUEST['redirect_to'] ) ) : admin_url(),
				'translatables' => apply_filters( 'rsssl_two_factor_translatables', [] ),
				'user_id'       => $verified_user_id
			] );
		}
	}

	/**
	 * Fallback enqueue script for browsers that do not support module scripts
	 *
	 * @param string $uri
	 * @param string $file_path
	 *
	 * @return void
	 */
	private function fallback_enqueue_script( string $uri, string $file_path ): void {
		wp_enqueue_script( 'rsssl-passkey-login', $uri, [], filemtime( $file_path ), true );
		add_filter( 'script_loader_tag', static function ( $tag, $handle ) {
			if ( $handle !== 'rsssl-passkey-login' ) {
				return $tag;
			}

			return str_replace( ' src', ' type="module" src', $tag );
		}, 10, 2 );
	}

	/**
	 * Starts the passkey onboarding process if the user has requested it.
	 *
	 * @param string $user_login The user's login name.
	 * @param WP_User $user The user object.
	 *
	 * @return void
	 * @throws \Exception
	 */
	public function maybe_start_passkey_onboarding( string $user_login, WP_User $user ): void {
		// Get the passkey status from user meta.
		$passkey_status = get_user_meta( $user->ID, 'rsssl_passkey_configured', true );
		// If passkey is already configured or ignored, do not start onboarding.
		if ( in_array( $passkey_status, [ 'configured', 'ignored' ], true ) ) {
			return;
		}

		// Otherwise, start the passkey onboarding.
		self::$onboarding_user = $user;
		wp_clear_auth_cookie();
		$this->passkey_onboarding_html( $user );
		exit;
	}

	/**
	 * Render onboarding UI and halt execution.
	 *
	 * @throws \Exception if nonce generation fails
	 */
	private function passkey_onboarding_html( WP_User $user ): void {
		// Variables needed for the template and scripts
		$redirect_to = admin_url();
//		$login_nonce = Rsssl_Two_Fa_Authentication::create_login_nonce( $user->ID )['rsssl_key'];

		add_action( 'login_enqueue_scripts', [ $this, 'enqueue_onboarding_scripts' ] );

		// Ensure login_header and login_footer functions are available
		if ( ! function_exists( 'login_header' ) ) {
			include_once rsssl_path . 'security/wordpress/two-fa/function-login-header.php';
		}

		if ( ! function_exists( 'login_footer' ) ) {
			include_once rsssl_path . 'security/wordpress/two-fa/function-login-footer.php';
		}

		//Add the styles for the two-factor authentication.
		add_action( 'login_enqueue_scripts', [ Rsssl_Two_Factor::class, 'enqueue_onboarding_styles' ] );

		login_header(
			__( 'Passkey Setup', 'really-simple-ssl' ),
			'',
			null
		);

		rsssl_load_template(
			'onboarding.php',
			[
				'user'        => $user,
				'redirect_to' => $redirect_to,
			//	'login_nonce' => $login_nonce,
			],
			rsssl_path . 'pro/assets/templates/passkey/'
		);

		login_footer();

		if ( ob_get_level() > 0 ) {
			ob_flush();
		}
		flush();
		exit; //This was the original exit.
	}
}

Rsssl_Passkey::run_hooks();